<?php if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-project-innovation-initiative
* @subpackage controllers
* @filesource
*/ /** */

require_once APPPATH.'controllers/inbox.php';

/**
* @package direct-project-innovation-initiative
* @subpackage controllers
*/
class Folder_controller extends Inbox {

    /** Archives all messages in a folder, then deletes the folder. */
    public function archive_folder($folder_name) {
		//make sure we have a folder name 
		$folder_name = base64_decode(rawurldecode($folder_name));
		if(!$this->is->nonempty_string($folder_name)) { 
			if(is_on_local()) $this->error->should_be_a_folder_name($folder_name);
			show_404(); 
		}
		$folder_name = CUSTOM_MAILBOX_PREFIX.$folder_name; //(add custom mailbox prefix)
		
		//make sure this is an existing custom folder
		$folder = Folder::find_by_name($folder_name);
		if(!Folder::is_an_entity($folder) || !$folder->is_custom_folder()) { 
			if(is_on_local()) $this->error->should_be_a_custom_folder_entity($folder);
			show_404(); 
		}

		//actually delete the folder
		if(Folder::delete($folder->id))
			$this->session->set_success_message($folder->name_for_display().' folder has been successfully archived.');
		else
			$this->session->set_error_message('An error occured and the '.$folder->name_for_display().' folder was not archived.  Please try again in a moment, and contact an administrator if this problem persists.');
        
        redirect('inbox');
    }
	
	
	/** Displays the create folder form view, meant for use with a modal window or pop-up. */
	public function create_folder_form() {
		$this->load->view('inbox/create_folder_form', array('custom_folder_data' => $this->mailformat->get_custom_folder_data(null)));
	}
	
	/**
	* Change the parent folder for a given folder.
	* Folder data is provided via POST.  Note that this method is used both by AJAX calls and by normal POST calls.  
	* @todo It would be simpler & easier to avoid bugs if we identified both folders by id instead of name.
	*/
    public function change_parent() {
		
		//make sure that a folder name has been provided		
		$folder_name = base64_decode(urldecode($this->input->post('encoded_folder_name',TRUE)));		
		if(!$this->is->nonempty_string($folder_name)){
			if(is_on_local()) $this->error->should_be_a_folder_name($folder_name);
			show_404();		
		}
		$folder_name = CUSTOM_MAILBOX_PREFIX.$folder_name; //add the custom prefix after verifying that the string isn't empty - otherwise, it will never be empty
		
		//make sure that a target parent has been provided and belongs to a folder in this mailbox
    	$parent_id = $this->input->post('parent_folder_select',TRUE);		
		if(empty($parent_id)) $parent_id = null;
		if(!is_null($parent_id) && !Folder::formatted_like_an_id($parent_id)){
			if(is_on_local()) $this->error->should_be_null_or_a_folder_id($parent_id);
			show_404();
		}
				
		//make sure that the folder name belongs to a folder in this mailbox
		$folder = Folder::find_by_name($folder_name);
		if(!Folder::is_an_entity($folder)){
			if(is_on_local()) $this->error->should_be_the_name_of_an_existing_folder($folder_name);
			show_404();
		}
    	
		//make the change, and report the result appropriately depending on whether or not this is an AJAX call
		$status = $folder->change_parent($parent_id);
		if($status){
    		$message = 'Successfully changed the '.$folder->name_for_display().' folder.';
    		if(!IS_AJAX) $this->session->set_success_message($message);
    	}
    	else {
    		$message = 'An error occured and the '. $folder->name_for_display() .' folder did not change its parent.  '.
					   'Please try again in a moment, and contact an administrator if this problem persists.';
    		if(!IS_AJAX) $this->session->set_error_message($message);
    	}
    	
		if(!IS_AJAX) redirect('inbox'); 
		echo $this->json->encode(compact('status', 'message'));
    }		
	
#TODO  - we should really be looking up folders by id, not by name -- much less potential for problems.	
	public function change_parent_form() {
		$data['custom_folder_data'] = $this->mailformat->get_custom_folder_data($this->input->get('encoded_folder_name',TRUE));
		$data['encoded_folder_name'] = $this->input->get('encoded_folder_name',TRUE);
		$data['folder_name'] = base64_decode(urldecode($this->input->get('encoded_folder_name',TRUE)));
		$data['parent_folder_name'] = $this->input->get('parent_folder_name',TRUE);
		$data['grand_parent_container_id'] = $this->input->get('grand_parent_container_id',TRUE);
		$data['deepest_descendant_level'] = $this->input->get('deepest_descendant_level',TRUE);
		$this->load->view('inbox/change_parent_form', $data);
	}
    
     /**
	 * Renames custom folders that the user has created with this application.
     * The function will check for the prefix that the application attaches to custom folders so that it replaces only
     * custom folders. It accepts the new folder name as a post value from the jeditable editable input and the old folder name
     * as a parameter passed to the function.
     */
#TODO - we should really be looking up folders by id, not by name -- much less potential for problems.	
    function rename_folder($folder_id, $folder_name, $show_unseen) {
		if(!Folder::formatted_like_an_id($folder_id)) show_404();
		if(!$this->is->nonempty_string($folder_name)) show_404();
		$folder_name = base64_decode(html_entity_decode(rawurldecode($folder_name)));
		$folder_info = array('folder_id'=>$folder_id, 'folder_name'=>$folder_name, 'show_unseen'=>$show_unseen);
		//make sure a new folder name has been specified		
		$new_folder_name = $this->input->post('value',TRUE);	
		if(empty($new_folder_name)) $new_folder_name = $this->input->post('value_'.hash('sha256',$folder_str),TRUE);
		if(empty($new_folder_name)){
			if(is_on_local()) $this->error->warning('No new folder name provided for '.$folder_name);
			show_404();
		}
		
		//make sure we're dealing with an existing custom folder
		$folder =  Folder::find($folder_id)[$folder_id];
		if(!Folder::is_an_entity($folder) || !$folder->is_custom_folder()){
			/* this original code blows the folder name part of the page if $folder can not be retrieved
			if(is_on_local()) $this->error->should_be_the_name_of_a_custom_folder($folder_name);
			show_404(); */
			echo $this->build_rename_response('failure', $folder, $folder_info, $new_folder_name);
			return;
		}
		
		//if this folder already has that name, we're done
		if($folder->name_for_display == $new_folder_name){
			echo $this->build_rename_response('success', $folder, $folder_info, $new_folder_name);
			return;
		}
		
		if(Folder::is_an_entity(Folder::find_by_name(CUSTOM_MAILBOX_PREFIX.$new_folder_name))){
			echo $this->build_rename_response('exists', $folder, $folder_info, $new_folder_name);
			return;
		}
		
		if($folder->rename($new_folder_name) && $folder->name_for_display == $new_folder_name){
			echo $this->build_rename_response('success', $folder, $folder_info, $new_folder_name);
			return;
		}
		
		echo $this->build_rename_response('failure', $folder, $folder_info, $new_folder_name);
	}

////////////////////////////////////
// PROTECTED HELPER METHODS
////////////////////////////////////

	protected function build_rename_response($response_type, $folder, $folder_info, $new_folder_str){		
		
		$folder_str_id = hash('sha256',$folder_info['folder_name']);
		$js_safe_new_folder_str = str_replace("'","\'",$new_folder_str);
		$js_safe_folder_str = str_replace("'","\'",$folder_info['folder_name']);
		if($response_type === 'success') {
			$show_unseen = ($folder->new > 0) ? ' ('.$folder->new.')' : ''; //would be better to just pass the folder object to this method
			return $new_folder_str .
				"<script>" .
					"$('#' + '" . $folder_str_id . "').parent().children('.custom_folder_name').css('display','none'); ". //hide the span used for edit
					"$('#anchor_' + '" . $folder->id . "').attr('title', '" . $new_folder_str . $show_unseen . "'); ". //change the title tooltip to new mailbox name
					"$('#anchor_' + '" . $folder->id . "').attr('href', '".site_url('inbox/change_mailbox/' . rawurlencode(base64_encode($folder->id)))."'); ". //change link href to new mailbox link
					"$('#anchor_' + '" . $folder->id . "').html('".$js_safe_new_folder_str . $show_unseen."'); " . //change link text to new mailbox name
					"$('#input_span_' + '" . $folder->id . "').html('".$js_safe_new_folder_str."');". //ensure hidden span has only the actual name
					"$('#anchor_' + '" . $folder->id . "').css('display', 'block'); ". //show the link again now that it is updated
					"$('#anchor_' + '" . $folder->id . "').focus(); ".//focus on after change
					"$('#' + '" . $folder_str_id . "').attr('id','" . hash('sha256',$new_folder_str) . "'); ". //set the id of the overall list item that associates all these elements with the mailbox name
					//reset the onclick event for the hidden button in the accessibility menu to use new mailbox
					"$('#rename_folder_" . $folder_str_id . "').attr('onclick','".
						"$(\'[id=\"".hash('sha256',$new_folder_str)."\"]\').parent().children(\'.custom_folder_name\').css(\'display\', \'inline\');".
						"$(\'[id=\"".hash('sha256',$new_folder_str)."\"]\').parent().children(\'a\').css(\'display\', \'none\');".
						"$(\'[id=\"".hash('sha256',$new_folder_str)."\"]\').parent().children(\'.custom_folder_name\').editable(\'/inbox/rename_folder/".
						rawurlencode(base64_encode($new_folder_str))."/\', { event : \'rename_event\', height: 12, width: 100, style : \'position: relative; float: right; z-index: 100;\', onblur : \'submit\',indicator : \'Saving...\', callback: function(value, settings) { $(this).unbind(settings.event);}}); ".
						"$(\'[id=\"".hash('sha256',$new_folder_str)."\"]\').parent().children(\'.custom_folder_name\').trigger(\'rename_event\');".
					"');".
					"$('#rename_folder_" . $folder_str_id . "').attr('id', 'rename_folder_" . hash('sha256',$new_folder_str) . "'); ". //set the id of the button to use new mailbox hash
					"$('#archive_folder_" . $folder_str_id . "').attr('href', '/inbox/archive_folder/".rawurlencode(base64_encode($new_folder_str))."/'); ". //set the hidden archive folder link to use new mailbox
					"$('#archive_folder_" . $folder_str_id . "').html('Archive Folder: " . ($new_folder_str) . "'); ". //set hidden archive folder link to display new mailbox name
					"$('#archive_folder_" . $folder_str_id . "').attr('id', 'archive_folder_" . hash('sha256',$new_folder_str) . "'); ". //set new id for archive folder link using new mailbox hash
					"$('#rename_folder_label_" . $folder_str_id . "').attr('for', 'rename_folder_" . hash('sha256',$new_folder_str) . "'); ". //set label for the hidden button to use new id
					"$('#rename_folder_label_" . $folder_str_id . "').attr('id', 'rename_folder_label_" . hash('sha256',$new_folder_str) . "'); ". //set id for the label to use the new mailbox hash
				"</script>"; 
		}
		else if($response_type === 'exists') {
			$show_unseen = ($folder->new > 0) ? ' ('.$folder->new.')' : ''; //would be better to just pass the folder object to this method
			return $folder_info['folder_name'] . $show_unseen . 
					"<script>".
						"$('#' + '" . $folder_str_id . "').parent().children('.custom_folder_name').css('display','none');". //hide the editable span
						"$('#anchor_' + '" . $folder->id . "').html('".$js_safe_folder_str . $show_unseen."');". //change the inner HTML of the link back to what it was
						"$('#' + '" . $folder_str_id . "').parent().children('.custom_folder_name').html('".$js_safe_folder_str."');". //ensure hidden span has only the actual name
						"$('#anchor_' + '" . $folder->id . "').css('display','block');". //show the link again
						"alert('A folder with this name already exists.');". //alert the user to the error
					"</script>";
		}
		else if($response_type === 'failure') {
			if($folder_info['show_unseen'] === '' || $folder_info['show_unseen'] === null) {
				$show_unseen = '';
			}
			else {
				$show_unseen = ' (' . $folder_info['show_unseen'] .')';
			}
			return $folder_info['folder_name'] . $show_unseen . 
					"<script>".
						"$('#' + '" . $folder_str_id . "').parent().children('.custom_folder_name').css('display','none');". //hide the editable span
						"$('#anchor_' + '" . $folder_info['folder_id'] . "').html('".$js_safe_folder_str . $show_unseen ."');". //change the inner HTML of the link back to what it was
						"$('#' + '" . $folder_str_id . "').parent().children('.custom_folder_name').html('".$js_safe_folder_str."');". //ensure hidden span has only the actual name
						"$('#anchor_' + '" . $folder_info['folder_id'] . "').css('display','block');". //show the link again
						"alert('An error occurred renaming folder, folder not renamed.');". //alert the user to the error
					"</script>";
		}
	}

	
}
/* End of file inbox.php */
/* Location: ./application/controllers/inbox.php */